#include "config.h"

#include <iostream>
#include <string>
#include <vector>

#include <dirent.h>
#include <fcntl.h>
#include <getopt.h>
#include <unistd.h>
#include <sys/stat.h>
#include <cstring>
#include <cstdlib>

#include "lib.misc.h"

#include "lib.error.hpp"

#define PROGRAM "evict"
#define VERSION DOMJUDGE_VERSION "/" REVISION

std::string_view progname;

bool be_verbose;
int show_help;
int show_version;
bool one_file_system;

struct option const long_opts[] = {
	{"verbose",         no_argument, nullptr,       'v'},
	{"one-file-system", no_argument, nullptr,       'x'},
	{"help",            no_argument, &show_help,     1 },
	{"version",         no_argument, &show_version,  1 },
	{ nullptr,          0,           nullptr,        0 }
};

void usage()
{
    std::cout << "Usage: " << progname << " [OPTION]... DIRECTORY" << std::endl
              << "Evicts all files in a directory tree from the kernel filesystem cache." << std::endl << std::endl
              << "  -v, --verbose          display some extra warnings and information" << std::endl
              << "  -x, --one-file-system  stay on this filesystem" << std::endl
              << "      --help             display this help and exit" << std::endl
              << "      --version          output version information and exit" << std::endl << std::endl;
	exit(0);
}

void evict_directory(const std::string& dirname, dev_t root_dev) {
	DIR *dir;
	struct dirent *entry;
	int fd = -1;
	struct stat s;

	dir = opendir(dirname.c_str());
	if (dir != nullptr) {
		if (be_verbose) logmsg(LOG_INFO, "Evicting all files in directory: {}", dirname);

		/* Read everything in the directory */
		while ( (entry = readdir(dir)) != nullptr ) {
			/* skip over current/parent directory entries */
			if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0) {
				continue;
			}

			/* Construct the full file path */
			std::string entry_path = dirname + "/" + entry->d_name;

			if (lstat(entry_path.c_str(), &s) < 0) {
				if (be_verbose) logerror(errno, "Unable to stat file/directory: {}", entry_path);
				continue;
			}

			if (one_file_system && s.st_dev != root_dev) {
				if (be_verbose) logmsg(LOG_DEBUG, "Skipping {}: different filesystem", entry_path);
				continue;
			}

			if (S_ISDIR(s.st_mode)) {
				/* Recurse into subdirectories */
				evict_directory(entry_path, root_dev);
			} else if (S_ISREG(s.st_mode)) {
				/* evict this file from the cache */
				fd = open(entry_path.c_str(), O_RDONLY, 0);
				if (fd == -1) {
					warning(errno, "Unable to open file: {}", entry_path);
					continue;
				}

				if (posix_fadvise(fd, 0, 0, POSIX_FADV_DONTNEED)) {
					warning(errno, "Unable to evict file: {}", entry_path);
				} else {
					if (be_verbose) logmsg(LOG_DEBUG, "Evicted file: {}", entry_path);
				}

				if ( close(fd)!=0 ) {
					warning(errno, "Unable to close file: {}", entry_path);
				}
			} else {
				/* We should never encounter other file types like symlinks,
				 * devices, or sockets in the judging directory, and they
				 * don't need to be evicted from the cache anyway. */
				if (be_verbose) logmsg(LOG_DEBUG, "Skipping {}: not a directory or regular file", entry_path);
			}
		}
		if ( closedir(dir)!=0 ) {
			warning(errno, "Unable to close directory: {}", dirname);
		}
	} else {
		warning(errno, "Unable to open directory: {}", dirname);
	}
}

int main(int argc, char *argv[])
{
	int opt;

	progname = argv[0];

	/* Parse command-line options */
	be_verbose = one_file_system = false;
	show_help = show_version = 0;
	opterr = 0;
	while ( (opt = getopt_long(argc,argv,"+vx",long_opts,nullptr))!=-1 ) {
		switch ( opt ) {
		case 0:   /* long-only option */
			break;
		case 'v': /* verbose option */
			be_verbose = true;
			verbose = LOG_DEBUG;
			break;
		case 'x': /* one-file-system option */
			one_file_system = true;
			break;
		case ':': /* getopt error */
		case '?':
			logmsg(LOG_ERR, "unknown option or missing argument `{}`", (char)optopt);
			break;
		default:
			logmsg(LOG_ERR, "getopt returned character code `{}` ??", (char)opt);
		}
	}

	if ( show_help ) usage();
	if ( show_version ) version(PROGRAM,VERSION);

	if ( argc<=optind ) {
		logmsg(LOG_ERR, "no directory specified");
		return 1;
	}

	/* directory to evict */
	std::string dirname = argv[optind];

	dev_t root_dev = 0;
	if (one_file_system) {
		struct stat s;
		if (stat(dirname.c_str(), &s) < 0) {
			logerror(errno, "Unable to stat directory: {}", dirname);
			return 1;
		}
		root_dev = s.st_dev;
	}

	evict_directory(dirname, root_dev);

	return 0;
}
