<?php declare(strict_types=1);

namespace App\Entity;

use App\Controller\API\AbstractRestController as ARC;
use App\DataTransferObject\ImageFile;
use App\DataTransferObject\TeamLocation;
use App\Repository\TeamRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use JMS\Serializer\Annotation as Serializer;
use OpenApi\Attributes as OA;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Context\ExecutionContextInterface;

/**
 * All teams participating in the contest.
 */
#[ORM\Entity(repositoryClass: TeamRepository::class)]
#[ORM\Table(options: ['collation' => 'utf8mb4_unicode_ci', 'charset' => 'utf8mb4'])]
#[ORM\Index(name: 'affilid', columns: ['affilid'])]
#[ORM\UniqueConstraint(name: 'externalid', columns: ['externalid'], options: ['lengths' => [190]])]
#[ORM\UniqueConstraint(name: 'label', columns: ['label'])]
#[UniqueEntity(fields: 'externalid')]
class Team extends BaseApiEntity implements
    HasExternalIdInterface,
    AssetEntityInterface,
    ExternalIdFromInternalIdInterface,
    PrefixedExternalIdInterface
{
    final public const DONT_ADD_USER = 'dont-add-user';
    final public const CREATE_NEW_USER = 'create-new-user';
    final public const ADD_EXISTING_USER = 'add-existing-user';

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column(options: ['comment' => 'Team ID', 'unsigned' => true])]
    #[Serializer\Exclude]
    protected ?int $teamid = null;

    #[ORM\Column(
        nullable: true,
        options: ['comment' => 'Team ID in an external system', 'collation' => 'utf8mb4_bin']
    )]
    #[Serializer\SerializedName('id')]
    protected ?string $externalid = null;

    #[ORM\Column(
        nullable: true,
        options: ['comment' => 'Team ID in the ICPC system', 'collation' => 'utf8mb4_bin']
    )]
    #[OA\Property(nullable: true)]
    #[Serializer\SerializedName('icpc_id')]
    protected ?string $icpcid = null;

    #[ORM\Column(
        nullable: true,
        options: ['comment' => 'Team label, for example the seat number', 'collation' => 'utf8mb4_bin']
    )]
    #[OA\Property(nullable: true)]
    protected ?string $label = null;

    #[ORM\Column(options: ['comment' => 'Team name', 'collation' => 'utf8mb4_bin'])]
    #[Assert\NotBlank]
    private string $name = '';

    #[ORM\Column(
        nullable: true,
        options: ['comment' => 'Team display name', 'collation' => 'utf8mb4_bin']
    )]
    #[OA\Property(nullable: true)]
    private ?string $display_name = null;

    #[ORM\Column(options: [
        'comment' => 'Whether the team is visible and operational',
        'default' => 1,
    ])]
    #[Serializer\Exclude]
    private bool $enabled = true;

    #[ORM\Column(
        name: 'publicdescription',
        type: 'text',
        nullable: true,
        options: ['comment' => 'Public team definition; for example: Team member names (freeform)']
    )]
    #[OA\Property(nullable: true)]
    #[Serializer\Groups([ARC::GROUP_NONSTRICT])]
    private ?string $publicDescription = null;

    #[ORM\Column(nullable: true, options: ['comment' => 'Physical location of team'])]
    #[OA\Property(nullable: true)]
    #[Serializer\Exclude]
    private ?string $location = null;

    #[ORM\Column(
        name: 'internalcomments',
        type: 'text',
        nullable: true,
        options: ['comment' => 'Internal comments about this team (jury only)']
    )]
    #[Serializer\Exclude]
    private ?string $internalComments = null;

    #[ORM\Column(
        type: 'decimal',
        precision: 32,
        scale: 9,
        nullable: true,
        options: ['comment' => 'Start time of last judging for prioritization', 'unsigned' => true]
    )]
    #[Serializer\Exclude]
    private string|float|null $judging_last_started = null;

    #[ORM\Column(options: ['comment' => 'Additional penalty time in minutes', 'default' => 0])]
    #[Serializer\Exclude]
    private int $penalty = 0;

    #[Serializer\Exclude]
    private string $addUserForTeam = self::DONT_ADD_USER;

    #[Assert\Regex(pattern: '/^[a-z0-9@._-]+$/i', message: 'Only alphanumeric characters and _-@. are allowed')]
    #[Serializer\Exclude]
    private ?string $newUsername = null;

    #[Serializer\Exclude]
    private ?User $existingUser = null;

    #[Assert\File(mimeTypes: ['image/png', 'image/jpeg', 'image/svg+xml'], mimeTypesMessage: "Only PNG's, JPG's and SVG's are allowed")]
    #[Serializer\Exclude]
    private ?UploadedFile $photoFile = null;

    #[Serializer\Exclude]
    private bool $clearPhoto = false;

    #[ORM\ManyToOne(inversedBy: 'teams')]
    #[ORM\JoinColumn(name: 'affilid', referencedColumnName: 'affilid', onDelete: 'SET NULL')]
    #[Serializer\Exclude]
    private ?TeamAffiliation $affiliation = null;

    /**
     * @var Collection<int, TeamCategory>
     */
    #[ORM\ManyToMany(targetEntity: TeamCategory::class, mappedBy: 'teams', cascade: ['persist'])]
    #[Serializer\Exclude]
    private Collection $categories;

    /**
     * @var Collection<int, Contest>
     */
    #[ORM\ManyToMany(targetEntity: Contest::class, mappedBy: 'teams')]
    #[Serializer\Exclude]
    private Collection $contests;

    /**
     * @var Collection<int, User>
     */
    #[ORM\OneToMany(targetEntity: User::class, mappedBy: 'team', cascade: ['persist'])]
    #[Assert\Valid]
    #[Serializer\Exclude]
    private Collection $users;

    /**
     * @var Collection<int, Submission>
     */
    #[ORM\OneToMany(targetEntity: Submission::class, mappedBy: 'team')]
    #[Serializer\Exclude]
    private Collection $submissions;

    /**
     * @var Collection<int, Clarification>
     */
    #[ORM\OneToMany(targetEntity: Clarification::class, mappedBy: 'sender')]
    #[Serializer\Exclude]
    private Collection $sent_clarifications;

    /**
     * @var Collection<int, Clarification>
     */
    #[ORM\OneToMany(targetEntity: Clarification::class, mappedBy: 'recipient')]
    #[Serializer\Exclude]
    private Collection $received_clarifications;

    /**
     * @var Collection<int, Clarification>
     */
    #[ORM\ManyToMany(targetEntity: Clarification::class)]
    #[ORM\JoinTable(name: 'team_unread')]
    #[ORM\JoinColumn(name: 'teamid', referencedColumnName: 'teamid', onDelete: 'CASCADE')]
    #[ORM\InverseJoinColumn(name: 'mesgid', referencedColumnName: 'clarid', onDelete: 'CASCADE')]
    #[Serializer\Exclude]
    private Collection $unread_clarifications;

    // This field gets filled by the team visitor with a data transfer
    // object that represents the team photo
    #[Serializer\Exclude]
    private ?ImageFile $photoForApi = null;

    public function setTeamid(int $teamid): Team
    {
        $this->teamid = $teamid;
        return $this;
    }

    public function getTeamid(): ?int
    {
        return $this->teamid;
    }

    public function setExternalid(?string $externalid): Team
    {
        $this->externalid = $externalid;
        return $this;
    }

    public function getExternalid(): ?string
    {
        return $this->externalid;
    }

    public function setIcpcid(?string $icpcid): Team
    {
        $this->icpcid = $icpcid;

        return $this;
    }

    public function getIcpcId(): ?string
    {
        return $this->icpcid;
    }

    public function setLabel(?string $label): Team
    {
        $this->label = $label;
        return $this;
    }

    public function getLabel(): ?string
    {
        return $this->label;
    }

    public function setName(string $name): Team
    {
        $this->name = $name;
        return $this;
    }

    public function getName(): string
    {
        return $this->name;
    }

    public function setDisplayName(?string $display_name): self
    {
        $this->display_name = $display_name;
        return $this;
    }

    public function getDisplayName(): ?string
    {
        return $this->display_name;
    }

    public function getEffectiveName(): string
    {
        return $this->getDisplayName() ?? $this->getName();
    }

    public function getShortDescription(): string
    {
        return $this->getEffectiveName();
    }

    public function setEnabled(bool $enabled): Team
    {
        $this->enabled = $enabled;
        return $this;
    }

    public function getEnabled(): bool
    {
        return $this->enabled;
    }

    public function setPublicDescription(?string $publicDescription): Team
    {
        $this->publicDescription = $publicDescription;
        return $this;
    }

    public function getPublicDescription(): ?string
    {
        return $this->publicDescription;
    }

    public function setLocation(?string $location): Team
    {
        $this->location = $location;
        return $this;
    }

    public function getLocation(): ?string
    {
        return $this->location;
    }

    #[Serializer\Groups([ARC::GROUP_NONSTRICT])]
    #[Serializer\VirtualProperty]
    #[Serializer\SerializedName('location')]
    public function getLocationForApi(): ?TeamLocation
    {
        return $this->location ? new TeamLocation($this->location) : null;
    }

    public function setInternalComments(?string $comments): Team
    {
        $this->internalComments = $comments;
        return $this;
    }

    public function getInternalComments(): ?string
    {
        return $this->internalComments;
    }

    public function setJudgingLastStarted(string|float $judgingLastStarted): Team
    {
        $this->judging_last_started = $judgingLastStarted;
        return $this;
    }

    public function getJudgingLastStarted(): string|float
    {
        return $this->judging_last_started;
    }

    public function setPenalty(int $penalty): Team
    {
        $this->penalty = $penalty;
        return $this;
    }

    /**
     * Set whether to add a new user for this team, link an existing one or do nothing.
     * Will not be stored, but is used in validation.
     */
    public function setAddUserForTeam(string $addUserForTeam): void
    {
        $this->addUserForTeam = $addUserForTeam;
    }

    /**
     * Set the username of a new user to add when $addUserForTeam is
     * static::CREATE_NEW_USER
     * Will not be stored, but is used in validation.
     */
    public function setNewUsername(?string $newUsername): Team
    {
        $this->newUsername = $newUsername;
        return $this;
    }

    /**
     * Set the user to link when $addUserForTeam is
     * static::ADD_EXISTING_USER
     * Will not be stored, but is used in validation.
     */
    public function setExistingUser(?User $existingUser): Team
    {
        $this->existingUser = $existingUser;
        return $this;
    }

    public function getPenalty(): int
    {
        return $this->penalty;
    }

    public function getAddUserForTeam(): string
    {
        return $this->addUserForTeam;
    }

    public function getNewUsername(): ?string
    {
        return $this->newUsername;
    }

    public function getExistingUser(): ?User
    {
        return $this->existingUser;
    }

    public function getPhotoFile(): ?UploadedFile
    {
        return $this->photoFile;
    }

    public function setPhotoFile(?UploadedFile $photoFile): Team
    {
        $this->photoFile = $photoFile;
        return $this;
    }

    public function isClearPhoto(): bool
    {
        return $this->clearPhoto;
    }

    public function setClearPhoto(bool $clearPhoto): Team
    {
        $this->clearPhoto = $clearPhoto;
        return $this;
    }

    public function setAffiliation(?TeamAffiliation $affiliation = null): Team
    {
        $this->affiliation = $affiliation;
        return $this;
    }

    public function getAffiliation(): ?TeamAffiliation
    {
        return $this->affiliation;
    }

    #[OA\Property(nullable: true)]
    #[Serializer\VirtualProperty]
    #[Serializer\SerializedName('organization_id')]
    public function getAffiliationId(): ?string
    {
        return $this->getAffiliation()?->getExternalid();
    }

    public function addCategory(TeamCategory $category): Team
    {
        $this->categories[] = $category;
        $category->addTeam($this);
        return $this;
    }

    public function removeCategory(TeamCategory $category): void
    {
        $this->categories->removeElement($category);
        $category->removeTeam($this);
    }

    /**
     * @return Collection<int, TeamCategory>
     */
    public function getCategories(): Collection
    {
        return $this->categories;
    }

    #[Serializer\VirtualProperty]
    #[Serializer\SerializedName('hidden')]
    #[Serializer\Type('bool')]
    public function getHidden(): bool
    {
        foreach ($this->getCategories() as $category) {
            if ($category->getVisible()) {
                return false;
            }
        }
        return true;
    }

    public function __construct()
    {
        $this->contests                = new ArrayCollection();
        $this->categories              = new ArrayCollection();
        $this->users                   = new ArrayCollection();
        $this->submissions             = new ArrayCollection();
        $this->sent_clarifications     = new ArrayCollection();
        $this->received_clarifications = new ArrayCollection();
        $this->unread_clarifications   = new ArrayCollection();
    }

    public function addContest(Contest $contest): Team
    {
        $this->contests[] = $contest;
        $contest->addTeam($this);
        return $this;
    }

    public function removeContest(Contest $contest): void
    {
        $this->contests->removeElement($contest);
        $contest->removeTeam($this);
    }

    /**
     * @return Collection<int, Contest>
     */
    public function getContests(): Collection
    {
        return $this->contests;
    }

    public function addUser(User $user): Team
    {
        $this->users[] = $user;
        $user->setTeam($this);
        return $this;
    }

    /**
     * @return Collection<int, User>
     */
    public function getUsers(): Collection
    {
        return $this->users;
    }

    public function addSubmission(Submission $submission): Team
    {
        $this->submissions[] = $submission;
        return $this;
    }

    /**
     * @return Collection<int, Submission>
     */
    public function getSubmissions(): Collection
    {
        return $this->submissions;
    }

    public function addSentClarification(Clarification $sentClarification): Team
    {
        $this->sent_clarifications[] = $sentClarification;
        return $this;
    }

    /**
     * @return Collection<int, Clarification>
     */
    public function getSentClarifications(): Collection
    {
        return $this->sent_clarifications;
    }

    public function addReceivedClarification(Clarification $receivedClarification): Team
    {
        $this->received_clarifications[] = $receivedClarification;
        return $this;
    }

    /**
     * @return Collection<int, Clarification>
     */
    public function getReceivedClarifications(): Collection
    {
        return $this->received_clarifications;
    }

    public function addUnreadClarification(Clarification $unreadClarification): Team
    {
        $this->unread_clarifications[] = $unreadClarification;
        return $this;
    }

    public function removeUnreadClarification(Clarification $unreadClarification): void
    {
        $this->unread_clarifications->removeElement($unreadClarification);
    }

    /**
     * @return Collection<int, Clarification>
     */
    public function getUnreadClarifications(): Collection
    {
        return $this->unread_clarifications;
    }

    /**
     * @return array{int}|array{}
     */
    #[Serializer\VirtualProperty]
    #[Serializer\Type('array<string>')]
    public function getGroupIds(): array
    {
        return $this->categories->map(fn(TeamCategory $category) => $category->getExternalid())->toArray();
    }

    #[OA\Property(nullable: true)]
    #[Serializer\VirtualProperty]
    #[Serializer\SerializedName('affiliation')]
    #[Serializer\Type('string')]
    #[Serializer\Groups([ARC::GROUP_NONSTRICT])]
    public function getAffiliationName(): ?string
    {
        return $this->getAffiliation()?->getName();
    }

    #[OA\Property(nullable: true)]
    #[Serializer\VirtualProperty]
    #[Serializer\Type('string')]
    #[Serializer\Groups([ARC::GROUP_NONSTRICT])]
    #[Serializer\Expose(if: "context.getAttribute('config_service').get('show_flags')")]
    public function getNationality() : ?string
    {
        return $this->getAffiliation()?->getCountry();
    }

    public function canViewClarification(Clarification $clarification): bool
    {
        return (($clarification->getSender() && $clarification->getSender()->getTeamid() === $this->getTeamid()) ||
            ($clarification->getRecipient() && $clarification->getRecipient()->getTeamid() === $this->getTeamid()) ||
            ($clarification->getSender() === null && $clarification->getRecipient() === null));
    }

    #[Assert\Callback]
    public function validate(ExecutionContextInterface $context): void
    {
        $this->validateUserCreation($context);
        $this->validateCategoryTypes($context);
    }

    private function validateUserCreation(ExecutionContextInterface $context): void
    {
        if ($this->getAddUserForTeam() === static::CREATE_NEW_USER) {
            if (empty($this->getNewUsername())) {
                $context
                    ->buildViolation('Required when adding a user')
                    ->atPath('newUsername')
                    ->addViolation();
            } elseif (!preg_match('/^[a-zA-Z0-9_-]+$/', $this->getNewUsername())) {
                $context
                    ->buildViolation('May only contain [a-zA-Z0-9_-].')
                    ->atPath('newUsername')
                    ->addViolation();
            }
        }
    }

    private function validateCategoryTypes(ExecutionContextInterface $context): void
    {
        $exclusiveTypes = [
            TeamCategory::TYPE_SCORING => 'scoring',
            TeamCategory::TYPE_BACKGROUND => 'background',
        ];
        $requiredTypes = [
            TeamCategory::TYPE_SCORING => 'scoring',
        ];

        foreach ($exclusiveTypes as $typeFlag => $typeName) {
            $categoriesWithType = [];
            foreach ($this->getCategories() as $category) {
                if ($category->hasType($typeFlag)) {
                    $categoriesWithType[] = $category->getName();
                }
            }

            $teamName = $this->getDisplayName() ?? $this->getName();
            if (isset($requiredTypes[$typeFlag]) && count($categoriesWithType) !== 1) {
                $context
                    ->buildViolation("Team $teamName must be in exactly one $typeName category")
                    ->atPath('categories')
                    ->addViolation();
            } elseif (!isset($requiredTypes[$typeFlag]) && count($categoriesWithType) > 1) {
                $context
                    ->buildViolation("Team $teamName can be in at most one $typeName category")
                    ->atPath('categories')
                    ->addViolation();
            }
        }
    }

    public function inContest(Contest $contest): bool
    {
        if ($contest->isOpenToAllTeams()) {
            return true;
        }
        if ($this->getContests()->contains($contest)) {
            return true;
        }
        foreach ($this->getCategories() as $category) {
            if ($category->inContest($contest)) {
                return true;
            }
        }
        return false;
    }

    public function getAssetProperties(): array
    {
        return ['photo'];
    }

    public function getAssetFile(string $property): ?UploadedFile
    {
        return match ($property) {
            'photo' => $this->getPhotoFile(),
            default => null,
        };
    }

    public function isClearAsset(string $property): ?bool
    {
        return match ($property) {
            'photo' => $this->isClearPhoto(),
            default => null,
        };
    }

    public function setPhotoForApi(?ImageFile $photoForApi = null): void
    {
        $this->photoForApi = $photoForApi;
    }

    /**
     * @return ImageFile[]
     */
    #[Serializer\VirtualProperty]
    #[Serializer\SerializedName('photo')]
    #[Serializer\Type('array<App\DataTransferObject\ImageFile>')]
    #[Serializer\Exclude(if: 'object.getPhotoForApi() === []')]
    public function getPhotoForApi(): array
    {
        return array_filter([$this->photoForApi]);
    }

    public function getCategoryOfType(int $type): ?TeamCategory
    {
        return $this->categories->findFirst(fn(int $key, TeamCategory $category) => $category->hasType($type));
    }

    /**
     * @return Collection<int, TeamCategory>
     */
    public function getCategoriesOfType(int $type): Collection
    {
        return $this->categories->filter(fn(TeamCategory $category) => $category->hasType($type));
    }

    public function getScoringCategory(): ?TeamCategory
    {
        return $this->getCategoryOfType(TeamCategory::TYPE_SCORING);
    }

    public function getBackgroundColorCategory(): ?TeamCategory
    {
        return $this->getCategoryOfType(TeamCategory::TYPE_BACKGROUND);
    }

    /**
     * @return Collection<int, TeamCategory>
     */
    public function getCssClassCategories(): Collection
    {
        return $this->getCategoriesOfType(TeamCategory::TYPE_CSS_CLASS);
    }

    /**
     * @return Collection<int, TeamCategory>
     */
    public function getTopBadgeCategories(): Collection
    {
        return $this->getCategoriesOfType(TeamCategory::TYPE_BADGE_TOP);
    }

    /**
     * @return Collection<int, TeamCategory>
     */
    public function getBadgeCategories(): Collection
    {
        return $this->getCategoriesOfType(TeamCategory::TYPE_BADGE_ALL);
    }

    public function getSortOrder(): ?int
    {
        return $this->getScoringCategory()?->getSortorder();
    }

    public function isLocked(): bool
    {
        foreach ($this->getContests() as $contest) {
            if ($contest->isLocked()) {
                return true;
            }
        }
        foreach ($this->getCategories() as $category) {
            foreach ($category->getContests() as $contest) {
                if ($contest->isLocked()) {
                    return true;
                }
            }
        }
        return false;
    }
}
